! All subroutines common to both programs except those that are part of
! the Tsyganenko model have been collected in this file.
! HG 2010-08-28
!
! I moved the bfield subroutine to this file, since it is not actually
! part of Tsyganenko's model; it only calls his subroutines.
! HG 2017-01-28
!
!------------------------------------------------------------------------


subroutine bfield(XGSW,YGSW,ZGSW,FXGSW,FYGSW,FZGSW,IOPT,PARMOD,EXNAME)

  dimension PARMOD(10)
  external EXNAME

  COMMON /GEOPACK1/ A(12),DS3,BB(2),PSI,CC(18)

  CALL EXNAME(IOPT,PARMOD,PSI,XGSW,YGSW,ZGSW,FXGSW,FYGSW,FZGSW)
  CALL IGRF_GSW_08(XGSW,YGSW,ZGSW,HXGSW,HYGSW,HZGSW)

  FXGSW = FXGSW + HXGSW
  FYGSW = FYGSW + HYGSW
  FZGSW = FZGSW + HZGSW

  return
end subroutine bfield

!------------------------------------------------------------------------

subroutine InitialiseParticles_01(vel,pos,X,Y,Z,vx0,vy0,vz0,vth,Lside,Npart)

! This subroutine initialises a particle distribution in a cube the
! side of which is Lside, and which is aligned with the axes of
! the cartesian coordinate system.


! No implicit variables
  implicit none

  integer Npart, pp

  real X, Y, Z, vx0, vy0, vz0

  real vel(Npart,3),pos(Npart,3), vth,Lside,rndp(3)

  do pp = 1, Npart
     call random_number(rndp)
     pos(pp,1) = X + rndp(1)*Lside - 0.5*Lside
     pos(pp,2) = Y + rndp(2)*Lside - 0.5*Lside
     pos(pp,3) = Z + rndp(3)*Lside - 0.5*Lside
  end do



! Use the subroutine Gaussian to produce thermal
! distributions.
! Maxwellian f(vx) ~ exp(-vx^2/vth^2) etc. => vth=sqrt(2*kB*T/m)

  call  Gaussian(vel,Npart,1,vx0,vth/sqrt(2.0))

  call  Gaussian(vel,Npart,2,vy0,vth/sqrt(2.0))

  call  Gaussian(vel,Npart,3,vz0,vth/sqrt(2.0))

  return
end subroutine InitialiseParticles_01

!------------------------------------------------------------------------

recursive subroutine InitialiseParticles_02(vel, pos, Npart, mass, &
     E, scdatafilename, &
     Nx, Ny, Nz, xmin, ymin, zmin, dxyz, Kp, parmod, attemptno, Nretries)

! No implicit variables
  implicit none

  integer Npart, Nx, Ny, Nz, Kp
  real Bgsw(3), B(3), B2, parmod(10)
  double precision mass, Eperp(3), E0(3), xmin, ymin, zmin, dxyz(3)
  double precision E(3,Nx,Ny,Nz)
  real vel(Npart,3), pos(Npart,3)
  character scdatafilename*200

  !particles data read from cluster data file
  double precision pos_gse(Npart,3), v_par(Npart), vth_perp(Npart)
  double precision indata(5), arg, NaN

  real rEarth, unitB(3), unitE(3), vdrift(3)
  integer pp, over, attemptno, Nretries

#ifdef _T01_
  external T01_01
#elif _TS05_
  external T04_s
#elif _T96_
  external T96_01
#endif

  parameter(rEarth = 6371.2e3) ! Must be the same as Tsyganenko's Earth radius
  scdatafilename = trim(scdatafilename)
  attemptno = attemptno + 1

  arg = -1.0d0
  NaN = sqrt(arg)

  ! Initialise variables
  pos = 0.0
  vel = 0.0
  indata = 0.0d0
  pos_gse = 0.0d0
  v_par = 0.0d0
  vth_perp = 0.0d0
  over = 0
  pp = 0

! Read cluster data file
  open(unit=1,file=scdatafilename,status='old',err=97)
  do while ((over==0) .and. (pp<Npart))
     read(1,*, end=95, err=99) indata
     pp = pp + 1
     pos_gse(pp,:) = indata(1:3)
     v_par(pp)     = indata(4)
     vth_perp(pp)  = indata(5)
  end do
95 over = 1
  close(1,err=99)

write(*,*) v_par(1:5), vth_perp(1:5)

! Set positions and velocities for each particle in GSW coordinates
  do pp = 1, Npart
     ! Convert the position we read in the file from GSE to GSW
     call gswgse_08(pos(pp,1), pos(pp,2), pos(pp,3), &
          real(pos_gse(pp,1)), real(pos_gse(pp,2)), real(pos_gse(pp,3)),-1)

     ! Find B field [nT] at that position (GSW)
#ifdef _T01_
     call bfield(pos(pp,1)/rEarth,pos(pp,2)/rEarth,pos(pp,3)/rEarth, &
          Bgsw(1),Bgsw(2),Bgsw(3),Kp,parmod,T01_01)
#elif _TS05_
     call bfield(pos(pp,1)/rEarth,pos(pp,2)/rEarth,pos(pp,3)/rEarth, &
          Bgsw(1),Bgsw(2),Bgsw(3),Kp,parmod,T04_s)
#elif _T96_
     call bfield(pos(pp,1)/rEarth,pos(pp,2)/rEarth,pos(pp,3)/rEarth, &
          Bgsw(1),Bgsw(2),Bgsw(3),Kp,parmod,T96_01)
#endif

     ! B in tesla instead of nanotesla; it is still in GSW though.
     B = Bgsw*1e-9

     ! squared magnitude of B
     B2 = B(1)**2.0d0 + B(2)**2.0d0 + B(3)**2.0d0

     ! unit vector in the same direction as Bgsw
     unitB = B/sqrt(B2)

     ! Find E field there too
     call InterpolateVectorField(E, Nx, Ny, Nz, dxyz, xmin, ymin, zmin, &
          dble(pos(pp,1)), dble(pos(pp,2)), dble(pos(pp,3)), E0)

     ! Compute perpendicular E, i.e. remove artificial parallel component
     call PerpendicularEfield(E0, Bgsw, Eperp)

     ! unit vector in the same direction as Eperp
     if (ALL(Eperp .EQ. [0.0d0,0.0d0,0.0d0]) .OR. ALL(isnan(Eperp))) then
        ! If the E field is 0, unitE is 0
        unitE = 0.0d0
        Eperp = 0.0d0
     else
        unitE = Eperp/sqrt(Eperp(1)**2.0d0 + Eperp(2)**2.0d0 + Eperp**2.0d0)
     end if

     ! Compute E cross B drift velocity
     vdrift(1) = ( Eperp(2)*B(3) - Eperp(3)*B(2) ) / B2
     vdrift(2) = ( Eperp(3)*B(1) - Eperp(1)*B(3) ) / B2
     vdrift(3) = ( Eperp(1)*B(2) - Eperp(2)*B(1) ) / B2

     vel(pp,1)=real(v_par(pp))*unitB(1)+vdrift(1) + real(vth_perp(pp))*unitE(1)
     vel(pp,2)=real(v_par(pp))*unitB(2)+vdrift(2) + real(vth_perp(pp))*unitE(2)
     vel(pp,3)=real(v_par(pp))*unitB(3)+vdrift(3) + real(vth_perp(pp))*unitE(3)

  end do

  return

! Error handling section
97 write (*,*) 'InitialiseParticles_02: error in open statement'
  goto 100
98 write (*,*) 'InitialiseParticles_02: error in read statement'
  close(1,err=99)
  goto 100
99 write (*,*) 'InitialiseParticles_02: error in close statement'
100 if (attemptno<=Nretries) then
     call InitialiseParticles_02(vel, pos, Npart, mass, E, &
         scdatafilename, Nx, Ny, Nz, &
         xmin, ymin, zmin, dxyz, Kp, parmod, attemptno, Nretries)
  end if

end subroutine InitialiseParticles_02

!------------------------------------------------------------------------

recursive subroutine InitialiseParticles_03(vel, pos, Npart, mass, &
     E, scdatafilename, &
     Nx, Ny, Nz, xmin, ymin, zmin, dxyz, Kp, parmod, attemptno, Nretries)

! No implicit variables
  implicit none

  integer Npart, Nx, Ny, Nz, Kp
  real Bgsw(3), B(3), B2, parmod(10)
  double precision mass, Eperp(3), E0(3), xmin, ymin, zmin, dxyz(3)
  double precision E(3,Nx,Ny,Nz)
  real vel(Npart,3), pos(Npart,3)
  character scdatafilename*200
  
  !particles data read from cluster data file  
  double precision pos_gse(Npart,3), v_par(Npart), v_perp(Npart)
  double precision indata(5)

  real rEarth, unitB(3), unitE(3)
  integer pp, over, attemptno, Nretries

#ifdef _T01_
  external T01_01
#elif _TS05_
  external T04_s
#elif _T96_
  external T96_01
#endif

  parameter(rEarth = 6371.2e3) ! Must be the same as Tsyganenko's Earth radius
  attemptno = attemptno + 1

  ! Initialise variables
  pos = 0.0
  vel = 0.0
  indata = 0.0d0
  pos_gse = 0.0d0
  v_par = 0.0d0
  v_perp = 0.0d0
  over = 0
  pp = 0

! Read cluster data file
  open(unit=1,file=scdatafilename,status='old',err=97)
  do while ((over==0) .and. (pp<Npart))
     read(1,*, end=95, err=99) indata
     pp = pp + 1
     pos_gse(pp,:) = indata(1:3)
     v_par(pp)     = indata(4)
     v_perp(pp)    = indata(5)
  end do
95 over = 1
  close(1,err=99)

! Set positions and velocities for each particle in GSW coordinates
  do pp = 1, Npart
     ! Convert the position we read in the file from GSE to GSW
     call gswgse_08(pos(pp,1), pos(pp,2), pos(pp,3), &
          real(pos_gse(pp,1)), real(pos_gse(pp,2)), real(pos_gse(pp,3)),-1)

     ! Find B field at that position
#ifdef _T01_
     call bfield(pos(pp,1)/rEarth,pos(pp,2)/rEarth,pos(pp,3)/rEarth, &
          Bgsw(1),Bgsw(2),Bgsw(3),Kp,parmod,T01_01)
#elif _TS05_
     call bfield(pos(pp,1)/rEarth,pos(pp,2)/rEarth,pos(pp,3)/rEarth, &
          Bgsw(1),Bgsw(2),Bgsw(3),Kp,parmod,T04_s)
#elif _T96_
     call bfield(pos(pp,1)/rEarth,pos(pp,2)/rEarth,pos(pp,3)/rEarth, &
          Bgsw(1),Bgsw(2),Bgsw(3),Kp,parmod,T96_01)
#endif

     ! B in tesla instead of nanotesla; it is still in GSW though.
     B = Bgsw*1e-9

     ! squared magnitude of B
     B2 = B(1)**2.0 + B(2)**2.0 + B(3)**2.0

     ! unit vector in the same direction as Bgsw
     unitB = B/sqrt(B2)

     ! Find E field there too
     call InterpolateVectorField(E, Nx, Ny, Nz, dxyz, xmin, ymin, zmin, &
          dble(pos(pp,1)), dble(pos(pp,2)), dble(pos(pp,3)), E0)

     ! Compute perpendicular E, i.e. remove artificial parallel component
     call PerpendicularEfield(E0, Bgsw, Eperp)

     ! unit vector in the same direction as Eperp
     if ( ALL(Eperp .NE. [0.0d0,0.0d0,0.0d0]) ) then
        unitE = real(Eperp/sqrt(Eperp(1)**2.0d0 + Eperp(2)**2.0d0 + &
             Eperp**2.0d0))
     else
        ! If the E field is 0, unitE is 0
        unitE = Eperp
     end if

     vel(pp,1)=real(v_par(pp))*unitB(1) + real(v_perp(pp))*unitE(1)
     vel(pp,2)=real(v_par(pp))*unitB(2) + real(v_perp(pp))*unitE(2)
     vel(pp,3)=real(v_par(pp))*unitB(3) + real(v_perp(pp))*unitE(3)

  end do

  return

! Error handling section
97 write (*,*) 'InitialiseParticles_02: error in open statement'
  goto 100
98 write (*,*) 'InitialiseParticles_02: error in read statement'
  close(1,err=99)
  goto 100
99 write (*,*) 'InitialiseParticles_02: error in close statement'
100 if (attemptno<=Nretries) then
     call InitialiseParticles_02(vel, pos, Npart, mass, E, scdatafilename, &
     Nx, Ny, Nz, xmin, ymin, zmin, dxyz, Kp, parmod, attemptno, Nretries)
  end if

end subroutine InitialiseParticles_03

!------------------------------------------------------------------------

subroutine Gaussian(x,N,comp,mu,sigma)

! Generates a normally distributed sequence of random numbers
! f(v) ~ exp(-(v-mu)^2/(2*sigma^2)

! No implicit variables
  implicit none

! Parameters
  integer N, comp
  real x(N,3), mu, sigma

! Local variables
  integer ii
  real pi,rndp(2)
  pi = 3.14159265358979323846264338327950288419716d0

  do ii = 1, N
     call random_number(rndp)
     x(ii,comp) = sqrt(-2.0*log(rndp(1)))* &
          cos(2.0*pi*rndp(2))*sigma+mu
  end do

  return

end subroutine Gaussian


!------------------------------------------------------------------------


subroutine PerpendicularEfield(E, B, Eperp)

  ! No implicit variables

  implicit none
  real B(3)
  double precision E(3), Eperp(3)
  double precision u(3)

  ! Unit vector in the B direction
  u = dble(B) / sqrt(dble(B(1))**2.0d0+dble(B(2))**2.0d0+dble(B(3))**2.0d0)

  ! The component of E which is perpendicular to B
  if (ALL(isnan(E))) then
    Eperp = 0.0d0
  else
    Eperp = E - u*sum(u*E)
  end if

end subroutine PerpendicularEfield

!------------------------------------------------------------------------

subroutine Boris(velocity_in,velocity_out,E,B,dt,mass,charge)


! This is the velocity update. It uses the Boris algorithm as described
! by, e.g. Birdsall and Langdon.
!
!@Book{BL91,
!  author =	 {C. K. Birdsall and A. B. Langdon},
!  title =	 {Plasma physics via computer simulation},
!  publisher =	 "IOP Publishing Ltd",
!  address =	 "Dirac House, Temple Back, Bristol BS1 6BE, UK",
!  year =	 1991,
!  note =	 "{ISBN} 0 7503 1025 1"
!}
!
!PAGES:61-63
! No implicit variables
  implicit none

! Parameters
  real B(3)
  double precision velocity_in(3), velocity_out(3)
  double precision E(3)
  double precision dt, mass, charge


! Local variables
  double precision s(3), t(3), vplus(3), vmin(3), &
       vmid(3), eta, etadt2
  integer i


! Set constant local variables
  eta = charge/mass
  etadt2 = eta*dt/2.0d0


! Calculate t and vmin (first half acceleration)
  do i = 1, 3
     t(i) = etadt2*dble(B(i))
     vmin(i) = velocity_in(i)+etadt2*E(i)
  end do

! Cross product (vmid = vmin + vmin x t)
  vmid(1) = vmin(1) + vmin(2)*t(3)-vmin(3)*t(2)
  vmid(2) = vmin(2) + vmin(3)*t(1)-vmin(1)*t(3)
  vmid(3) = vmin(3) + vmin(1)*t(2)-vmin(2)*t(1)

! Calculate s
  do i = 1, 3
     s(i) = 2.0d0*t(i)/(1.0d0 + t(1)**2.0d0 + t(2)**2.0d0 + t(3)**2.0d0)
  end do

! Another cross product (vplus = vmin + vmid x s)
  vplus(1) = vmin(1) + vmid(2)*s(3)-vmid(3)*s(2)
  vplus(2) = vmin(2) + vmid(3)*s(1)-vmid(1)*s(3)
  vplus(3) = vmin(3) + vmid(1)*s(2)-vmid(2)*s(1)

! Loop and set new velocities (second half acceleration)
  do i= 1, 3
     velocity_out(i) = vplus(i) + etadt2*E(i)
  end do

  return
end subroutine Boris

!------------------------------------------------------------------------

subroutine SpeedMomentAngle(vv,mu,pangle,velocity,Bin,mass)

! No implicit variables
  implicit none
  real Bin(3)
  double precision mass
  double precision velocity(3), vv, vv2, mu, pangle

! local variables
  double precision bb, bb2, vperp2, B(3)

  B   = dble(Bin)                           ! convert to double precision
  bb2 = B(1)**2.0d0+B(2)**2.0d0+B(3)**2.0d0 ! magnitude of B squared
  bb  = sqrt(bb2)                           ! magnitude of B

  vv2= velocity(1)**2.0d0+velocity(2)**2.0d0+velocity(3)**2.0d0
  vv = sqrt(vv2)
  !vperp2 = |v|^2 - (vB)^2/|B|^2
  vperp2 = vv2 - (velocity(1)*B(1)+velocity(2)*B(2)+ &
       velocity(3)*B(3))**2.0d0/bb2
  mu=mass*vperp2/(2*bb)
  !angle = acos (vB/|v||B|) -> direction of vB
  pangle=acos((velocity(1)*B(1)+velocity(2)*B(2)+velocity(3)*B(3))/(vv*bb))

  return

end subroutine SpeedMomentAngle


!------------------------------------------------------------------------

recursive subroutine GetInput(iyear,iday,dhtime,pdyn,Dst, &
     ByIMF, BzIMF, Tsygvxgse, Tsygvygse, Tsygvzgse, &
     G1, G2, W1, W2, W3, W4, W5, W6, &
     Xgse, Ygse, Zgse, &
     vxgse, vygse, vzgse, Kp, Npart, Nsteps, vth, Lside, dt, mass, charge, &
     forward, &
     altobs, initialiser, scdatafilename, Nx, Ny, Nz, Parallel, &
     xmin, xmax, ymin, ymax, zmin, zmax, &
     msw, ALindex, UseAL, dumpfields, usedump, &
     number_of_steps, maximum_step_length, use_gyro_period, &
     use_max_step_length, saving_step_length, attemptno, Nretries)

  implicit none

  integer iyear, iday, Kp, Npart, Nsteps, initialiser, Nx, Ny, Nz, &
       number_of_steps, attemptno, Nretries
  real Xgse, Ygse, Zgse, pdyn, Dst, ByIMF, BzIMF, &
       Tsygvxgse, Tsygvygse, Tsygvzgse, &
       G1, G2, W1, W2, W3, W4, W5, W6, &
       vxgse, vygse, vzgse, dhtime, vth, Lside
  double precision dt, mass, charge, forward, &
          altobs, xmin, xmax, ymin, ymax, zmin, zmax
  double precision msw, ALindex, gLAT, gMLT, &
       maximum_step_length, saving_step_length
  logical UseAL, dumpfields, usedump, use_gyro_period, use_max_step_length
  ! use_gyro_period, use_max_step_length - added by KiAl
  character scdatafilename*200

  character indata*132, filename*242, Parallel*1
  integer i, j, k

  attemptno = attemptno + 1

 ! Defaults
  iyear  = 2000  ! Year for use with Tsyganenko
  iday   = 1     ! Day for use with Tsyganenko
  dhtime = 1.0   ! Decimal hours for use with Tsyganenko
  Dst    = -2.0e2     ! The Dst index [nT]
  pdyn   = 2.0e1      ! Solar wind dynamic pressure [nPa]
  ByIMF  = 2.0e1      ! IMF By component for use with Tsyganenko [nT]
  BzIMF  = -1.0e1     ! IMF Bz component for use with Tsyganenko [nT]
  Tsygvxgse  = -400.0 ! x-component [km/s] of bulk velocity in GSE system
  Tsygvygse  = 0.0    ! y-component [km/s] of bulk velocity in GSE system
  Tsygvzgse  = 0.0    ! z-component [km/s] of bulk velocity in GSE system
  G1 = 6.0            ! Parameter for the T01 model
  G2 = 10.0           ! Parameter for the T01 model
  W1 = 0.0            ! Parameter for the TS05 model
  W2 = 0.0            ! Parameter for the TS05 model
  W3 = 0.0            ! Parameter for the TS05 model
  W4 = 0.0            ! Parameter for the TS05 model
  W5 = 0.0            ! Parameter for the TS05 model
  W6 = 0.0            ! Parameter for the TS05 model
  Xgse   = 0.0   ! x-coordinate [m] in GSE system for centre of cube
  Ygse   = 0.0   ! y-coordinate [m] in GSE system for centre of cube
  Zgse   = 1.0e7 ! z-coordinate [m] in GSE system for centre of cube
  vxgse  = 0.0   ! x-component [m/s] of bulk velocity in GSE system
  vygse  = 0.0   ! y-component [m/s] of bulk velocity in GSE system
  vzgse  = 1.0e5 ! z-component [m/s] of bulk velocity in GSE system
  Kp     = 0     ! Kp-index  for use with Tsyganenko
  Npart  = 1     ! Number of trajectories to compute
  Nsteps = 10000 ! Maximum number of time steps per trajectory
  vth    = 1.0e5 ! Thermal speed [m/s] for Maxwellian
                 ! f(vx) ~ exp(-vx^2/vth^2) etc. => vth=sqrt(2*kB*T/m)
  Lside  = 1.0e6 ! Side of cube where particles start [m]
  dt     = 1.0d0 ! Time step [s]
  mass   = 2.67619d-26     ! Ion mass [kg]
  charge = 1.6021766208d-19! Ion charge [As]
  forward = 1.0e0;         ! To go forward in time: +1 to go backward: -1
  altobs = 1.0d6           ! Obstacle altitude
  initialiser = 1          ! Initialisation method
  scdatafilename='cluster.dat' ! Input file name for initialiser=2
  Nx = 171                 ! Number of E-field grid points in x-direction
  Ny = 21                  ! Number of E-field grid points in y-direction
  Nz = 11                  ! Number of E-field grid points in z-direction
  Parallel = 'x'           ! Direction in which we parallelize
  xmin = -1.019392e9       ! Lower x-boundary [m] of Efield grid, GSW system
  xmax =  6.3712e7         ! Upper x-boundary [m] of Efield grid, GSW system
  ymin = -6.3712e7         ! Lower y-boundary [m] of Efield grid, GSW system
  ymax =  6.3712e7         ! Upper y-boundary [m] of Efield grid, GSW system
  zmin =  0                ! Lower z-boundary [m] of Efield grid, GSW system
  zmax =  6.3712e7         ! Upper z-boundary [m] of Efield grid, GSW system
! Weimer model parameters
  msw = 1.8004e-27         ! msw: unit mass of the solar wind,
                           ! e.g., me+0.9231mp+0.0385malpha
  ALindex = 0.0e0          ! ALindex: (optional) AL index in nT
  UseAL   = .false.        ! If this is .true. ALindex is used, otherwise not.
  dumpfields = .false.     ! If this is .true. U and E are dumped after creation
  usedump =.false.         ! If this is .true. dumped fields are used

! Timestep calculation parameters - Added by KiAl
  number_of_steps = 20           ! number of steps in a gyroperiod
  maximum_step_length = 1000    ! maximum particle movement
  use_gyro_period = .false.     ! calculate timestep based on gyro period
  use_max_step_length = .false. ! calculate timestep based on particle
                                ! movement length

  filename= 'inputham.m'
  open(unit=1,file=filename,status='old',err=97)

  read (1,'(a)',err=98) indata

  do while (index(indata,'%END')+index(indata,'%end') == 0)
     i=index(indata,'=')
     if (i>0) then
        do j=1,i-1
           if(indata(j:j) /= ' ') exit
        enddo
        ! If there is a semicolon on the line, ignore it and everything beyond.
        k = index(indata(i+1:132),';')
        if (k==0) then
           k = 133-i
        end if
        select case (indata(j:i-1))
        case ('iyear')
           read (indata(i+1:i+k-1),*) iyear
        case ('iday')
           read (indata(i+1:i+k-1),*) iday
        case ('dhtime')
           read (indata(i+1:i+k-1),*) dhtime
        case ('Dst')
           read (indata(i+1:i+k-1),*) Dst
        case ('pdyn')
           read (indata(i+1:i+k-1),*) pdyn
        case ('ByIMF')
           read (indata(i+1:i+k-1),*) ByIMF
        case ('BzIMF')
           read (indata(i+1:i+k-1),*) BzIMF
        case ('Tsygvxgse')
           read (indata(i+1:i+k-1),*) Tsygvxgse
        case ('Tsygvygse')
           read (indata(i+1:i+k-1),*) Tsygvygse
        case ('Tsygvzgse')
           read (indata(i+1:i+k-1),*) Tsygvzgse
        case ('G1')
           read (indata(i+1:i+k-1),*) G1
        case ('G2')
           read (indata(i+1:i+k-1),*) G2
        case ('W1')
           read (indata(i+1:i+k-1),*) W1
        case ('W2')
           read (indata(i+1:i+k-1),*) W2
        case ('W3')
           read (indata(i+1:i+k-1),*) W3
        case ('W4')
           read (indata(i+1:i+k-1),*) W4
        case ('W5')
           read (indata(i+1:i+k-1),*) W5
        case ('W6')
           read (indata(i+1:i+k-1),*) W6
        case ('Xgse')
           read (indata(i+1:i+k-1),*) Xgse
        case ('Ygse')
           read (indata(i+1:i+k-1),*) Ygse
        case ('Zgse')
           read (indata(i+1:i+k-1),*) Zgse
        case ('vxgse')
           read (indata(i+1:i+k-1),*) vxgse
        case ('vygse')
           read (indata(i+1:i+k-1),*) vygse
        case ('vzgse')
           read (indata(i+1:i+k-1),*) vzgse
        case ('Kp')
           read (indata(i+1:i+k-1),*) Kp
        case ('Npart')
           read (indata(i+1:i+k-1),*) Npart
        case ('Nsteps')
           read (indata(i+1:i+k-1),*) Nsteps
        case ('vth')
           read (indata(i+1:i+k-1),*) vth
        case ('Lside')
           read (indata(i+1:i+k-1),*) Lside
        case ('dt')
           read (indata(i+1:i+k-1),*) dt
        case ('mass')
           read (indata(i+1:i+k-1),*) mass
        case ('charge')
           read (indata(i+1:i+k-1),*) charge
        case ('forward')
           read (indata(i+1:i+k-1),*) forward
        case ('altobs')
           read (indata(i+1:i+k-1),*) altobs
        case ('initialiser')
           read (indata(i+1:i+k-1),*) initialiser
        case ('scdatafilename')
           read (indata(i+1:i+k-1),*) scdatafilename
        case ('Nx')
           read (indata(i+1:i+k-1),*) Nx
        case ('Ny')
           read (indata(i+1:i+k-1),*) Ny
        case ('Nz')
           read (indata(i+1:i+k-1),*) Nz
       case ('Parallel')
           read (indata(i+1:i+k-1),*) Parallel
        case ('xmin')
           read (indata(i+1:i+k-1),*) xmin
        case ('xmax')
           read (indata(i+1:i+k-1),*) xmax
        case ('ymin')
           read (indata(i+1:i+k-1),*) ymin
        case ('ymax')
           read (indata(i+1:i+k-1),*) ymax
        case ('zmin')
           read (indata(i+1:i+k-1),*) zmin
        case ('zmax')
           read (indata(i+1:i+k-1),*) zmax
        case ('msw')
           read (indata(i+1:i+k-1),*) msw
        case ('ALindex')
           read (indata(i+1:i+k-1),*) ALindex
        case ('UseAL')
           if (index(indata(i+1:i+k-1),'on') > 0) then
              UseAL = .true.
           end if
        case ('dumpfields')
           if (index(indata(i+1:i+k-1),'yes') > 0) then
              dumpfields = .true.
           end if
        case ('usedump')
           if (index(indata(i+1:i+k-1),'yes') > 0) then
              usedump = .true.
           end if
        case ('number_of_steps')
           read (indata(i+1:i+k-1),*) number_of_steps
        case ('maximum_step_length')
           read (indata(i+1:i+k-1),*) maximum_step_length
        case ('saving_step_length')
           read (indata(i+1:i+k-1),*) saving_step_length
        case ('use_gyro_period')
           if (index(indata(i+1:i+k-1),'yes') > 0) then
              use_gyro_period = .true.
           end if
        case ('use_max_step_length')
           if (index(indata(i+1:i+k-1),'yes') > 0) then
              use_max_step_length = .true.
           end if
        case default
           if (index(indata(j:j+1),'%')==0) then
              write (*,*) 'Input quantity ', indata(j:i-1), ' is unknown.'
           end if
        end select
     end if
     read (1,'(a)',err=98) indata
  end do

  close(1,err=99)

  return

! Error handling section
97 write (*,*) 'GetInput: error in open statement'
  goto 100
98 write (*,*) 'GetInput: error in read statement'
  close(1,err=99)
  goto 100
99 write (*,*) 'GetInput: error in close statement'
100 if (attemptno<=Nretries) then
     call GetInput(iyear, iday, dhtime, pdyn, Dst, &
          ByIMF, BzIMF, Tsygvxgse, Tsygvygse, Tsygvzgse, &
          G1, G2, W1, W2, W3, W4, W5, W6, &
          Xgse, Ygse, Zgse, &
          vxgse, vygse, vzgse, Kp, Npart, Nsteps, vth, Lside, dt, &
          mass, charge, forward, &
          altobs, initialiser, scdatafilename, Nx, Ny, Nz, Parallel, &
          xmin, xmax, ymin, ymax, zmin, zmax, &
          msw, ALindex, UseAL, dumpfields, usedump, &
          number_of_steps, maximum_step_length, use_gyro_period, &
          use_max_step_length, saving_step_length, attemptno, Nretries)
  else
     stop
  end if

end subroutine GetInput


!------------------------------------------------------------------------

subroutine NegGradientOpen(U,E,Nx,Ny,Nz,dxyz)


! No implicit variables
  implicit none

! Parameters
  integer Nx, Ny, Nz
  double precision U(Nx,Ny,Nz), E(3,Nx,Ny,Nz), dxyz(3)

! Variables
  integer i, j, k, l
  double precision dx2, dy2, dz2, dx, dy, dz

! Set locals
  dx = dxyz(1)
  dy = dxyz(2)
  dz = dxyz(3)
  dx2 = dxyz(1)*2.0d0
  dy2 = dxyz(2)*2.0d0
  dz2 = dxyz(3)*2.0d0

! Interior points

  forall (k=2:Nz-1,j=2:Ny-1,i=2:Nx-1)
     E(1,i,j,k) = (U(i-1,j,k)-U(i+1,j,k))/dx2
     E(2,i,j,k) = (U(i,j-1,k)-U(i,j+1,k))/dy2
     E(3,i,j,k) = (U(i,j,k-1)-U(i,j,k+1))/dz2
  end forall

! Boundary points, open boundaries causes a little more
! elaborate procedure here than in the usual constant case.

! xy-plane
  k = 1
  do j = 2, Ny-1
     do i = 2, Nx-1
        E(1,i,j,k) = (U(i-1,j,k)-U(i+1,j,k))/dx2
        E(2,i,j,k) = (U(i,j-1,k)-U(i,j+1,k))/dy2
        E(3,i,j,k) = (U(i,j,k)-U(i,j,k+1))/dz
     end do
  end do

  k = Nz
  do j = 2, Ny-1
     do i = 2, Nx-1
        E(1,i,j,k) = (U(i-1,j,k)-U(i+1,j,k))/dx2
        E(2,i,j,k) = (U(i,j-1,k)-U(i,j+1,k))/dy2
        E(3,i,j,k) = (U(i,j,k-1)-U(i,j,k))/dz
     end do
  end do

! yz-plane
  i = 1
  do k = 2, Nz-1
     do j = 2, Ny-1
        E(1,i,j,k) = (U(i,j,k)-U(i+1,j,k))/dx
        E(2,i,j,k) = (U(i,j-1,k)-U(i,j+1,k))/dy2
        E(3,i,j,k) = (U(i,j,k-1)-U(i,j,k+1))/dz2
     end do
  end do

  i = Nx
  do k = 2, Nz-1
     do j = 2, Ny-1
        E(1,i,j,k) = (U(i-1,j,k)-U(i,j,k))/dx
        E(2,i,j,k) = (U(i,j-1,k)-U(i,j+1,k))/dy2
        E(3,i,j,k) = (U(i,j,k-1)-U(i,j,k+1))/dz2
     end do
  end do

! xz-plane
  j = 1
  do k = 2, Nz-1
     do i = 2, Nx-1
        E(1,i,j,k) = (U(i-1,j,k)-U(i+1,j,k))/dx2
        E(2,i,j,k) = (U(i,j,k)-U(i,j+1,k))/dy
        E(3,i,j,k) = (U(i,j,k-1)-U(i,j+1,k+1))/dz2
     end do
  end do

  j = Ny
  do k = 2, Nz-1
     do i = 2, Nx-1
        E(1,i,j,k) = (U(i-1,j,k)-U(i+1,j,k))/dx2
        E(2,i,j,k) = (U(i,j-1,k)-U(i,j,k))/dy
        E(3,i,j,k) = (U(i,j,k-1)-U(i,j,k+1))/dz2
     end do
  end do

! We need to take care of all the edges and corners too.
! So, here begins the really boring part.

! All edges along x

! z = y = 0
  j = 1
  k = 1
  do i = 2, Nx-1
     E(1,i,j,k) = (U(i-1,j,k)-U(i+1,j,k))/dx2
     E(2,i,j,k) = (U(i,j,k)-U(i,j+1,k))/dy
     E(3,i,j,k) = (U(i,j,k)-U(i,j,k+1))/dz
  end do

! z = zmax, y = 0
  k = Nz
  do i = 2, Nx-1
     E(1,i,j,k) = (U(i-1,j,k)-U(i+1,j,k))/dx2
     E(2,i,j,k) = (U(i,j,k)-U(i,j+1,k))/dy
     E(3,i,j,k) = (U(i,j,k-1)-U(i,j,k))/dz
  end do


! z = zmax, y = ymax
  j = Ny
  do i = 2, Nx-1
     E(1,i,j,k) = (U(i-1,j,k)-U(i+1,j,k))/dx2
     E(2,i,j,k) = (U(i,j-1,k)-U(i,j,k))/dy
     E(3,i,j,k) = (U(i,j,k-1)-U(i,j,k))/dz
  end do

! z = 0, y = ymax
  k = 1
  do i = 2, Nx-1
     E(1,i,j,k) = (U(i-1,j,k)-U(i+1,j,k))/dx2
     E(2,i,j,k) = (U(i,j-1,k)-U(i,j,k))/dy
     E(3,i,j,k) = (U(i,j,k)-U(i,j,k+1))/dz
  end do


! All edges along y

! x = z = 0
  i = 1
  k = 1
  do j = 2, Ny-1
     E(1,i,j,k) = (U(i,j,k)-U(i+1,j,k))/dx
     E(2,i,j,k) = (U(i,j-1,k)-U(i,j+1,k))/dy2
     E(3,i,j,k) = (U(i,j,k)-U(i,j,k+1))/dz
  end do

! x = xmax, z = 0
  i = Nx
  do j = 2, Ny-1
     E(1,i,j,k) = (U(i-1,j,k)-U(i,j,k))/dx
     E(2,i,j,k) = (U(i,j-1,k)-U(i,j+1,k))/dy2
     E(3,i,j,k) = (U(i,j,k)-U(i,j,k+1))/dz
  end do

! x = xmax, z = zmax
  k = Nz
  do j = 2, Ny-1
     E(1,i,j,k) = (U(i-1,j,k)-U(i,j,k))/dx
     E(2,i,j,k) = (U(i,j-1,k)-U(i,j+1,k))/dy2
     E(3,i,j,k) = (U(i,j,k-1)-U(i,j,k))/dz
  end do

! x = 0, z = zmax
  i = 1
  do j = 2, Ny-1
     E(1,i,j,k) = (U(i,j,k)-U(i+1,j,k))/dx
     E(2,i,j,k) = (U(i,j-1,k)-U(i,j+1,k))/dy2
     E(3,i,j,k) = (U(i,j,k-1)-U(i,j,k))/dz
  end do

! All edges along z

! x = y = 0
  i = 1
  j = 1
  do k = 2, Nz-1
     E(1,i,j,k) = (U(i,j,k)-U(i+1,j,k))/dx
     E(2,i,j,k) = (U(i,j,k)-U(i,j+1,k))/dy
     E(3,i,j,k) = (U(i,j,k-1)-U(i,j,k+1))/dz2
  end do

! x = 0, y = ymax
  j = Ny
  do k = 2, Nz-1
     E(1,i,j,k) = (U(i,j,k)-U(i+1,j,k))/dx
     E(2,i,j,k) = (U(i,j-1,k)-U(i,j,k))/dy
     E(3,i,j,k) = (U(i,j,k-1)-U(i,j,k+1))/dz2
  end do

! x = xmax, y = ymax
  i = Nx
  do k = 2, Nz-1
     E(1,i,j,k) = (U(i-1,j,k)-U(i,j,k))/dx
     E(2,i,j,k) = (U(i,j-1,k)-U(i,j,k))/dy
     E(3,i,j,k) = (U(i,j,k-1)-U(i,j,k+1))/dz2
  end do

! x = xmax, y = 0
  j = 1
  do k = 2, Nz-1
     E(1,i,j,k) = (U(i-1,j,k)-U(i,j,k))/dx
     E(2,i,j,k) = (U(i,j,k)-U(i,j+1,k))/dy
     E(3,i,j,k) = (U(i,j,k-1)-U(i,j,k+1))/dz2
  end do

! Take care of all eight corners, don't worry it'll
! be over soon...

  i = 1
  j = 1
  k = 1
  E(1,i,j,k) = (U(i,j,k)-U(i+1,j,k))/dx
  E(2,i,j,k) = (U(i,j,k)-U(i,j+1,k))/dy
  E(3,i,j,k) = (U(i,j,k)-U(i,j,k+1))/dz
  i = Nx
  E(1,i,j,k) = (U(i-1,j,k)-U(i,j,k))/dx
  E(2,i,j,k) = (U(i,j,k)-U(i,j+1,k))/dy
  E(3,i,j,k) = (U(i,j,k)-U(i,j,k+1))/dz
  j = Ny
  E(1,i,j,k) = (U(i-1,j,k)-U(i,j,k))/dx
  E(2,i,j,k) = (U(i,j-1,k)-U(i,j,k))/dy
  E(3,i,j,k) = (U(i,j,k)-U(i,j,k+1))/dz
  k = Nz
  E(1,i,j,k) = (U(i-1,j,k)-U(i,j,k))/dx
  E(2,i,j,k) = (U(i,j-1,k)-U(i,j,k))/dy
  E(3,i,j,k) = (U(i,j,k-1)-U(i,j,k))/dz
  i = 1
  E(1,i,j,k) = (U(i,j,k)-U(i+1,j,k))/dx
  E(2,i,j,k) = (U(i,j-1,k)-U(i,j,k))/dy
  E(3,i,j,k) = (U(i,j,k-1)-U(i,j,k))/dz
  j = 1
  E(1,i,j,k) = (U(i,j,k)-U(i+1,j,k))/dx
  E(2,i,j,k) = (U(i,j,k)-U(i,j+1,k))/dy
  E(3,i,j,k) = (U(i,j,k-1)-U(i,j,k))/dz
  i = Nx
  E(1,i,j,k) = (U(i-1,j,k)-U(i,j,k))/dx
  E(2,i,j,k) = (U(i,j,k)-U(i,j+1,k))/dy
  E(3,i,j,k) = (U(i,j,k-1)-U(i,j,k))/dz
  i = 1
  j = Ny
  k = 1
  E(1,i,j,k) = (U(i,j,k)-U(i+1,j,k))/dx
  E(2,i,j,k) = (U(i,j-1,k)-U(i,j,k))/dy
  E(3,i,j,k) = (U(i,j,k)-U(i,j,k+1))/dz

  return
end subroutine NegGradientOpen

!------------------------------------------------------------------------
subroutine DivideRegion(Nx,Ny,Nz,Parallel,Nprocs,myid, &
        Nxstart,Nxend,Nystart,Nyend,Nzstart,Nzend)

    implicit none

    integer Nxi, Nprocs, myid, istart, iend
    integer Nxi_local, remainder

    integer Nx, Ny, Nz, &
        Nxstart,Nxend,Nystart,Nyend,Nzstart,Nzend

    character Parallel*1

    if (Parallel == 'x') then
        Nxi = Nx
    elseif (Parallel == 'y') then
        Nxi = Ny
    elseif (Parallel == 'z') then
        Nxi = Nz
    endif

    Nxi_local = Nxi/Nprocs
    istart = myid*Nxi_local + 1
    remainder = mod(Nxi,Nprocs)
    istart = istart + min(myid,remainder)

    if (myid < remainder) then
        Nxi_local = Nxi_local + 1
    end if
    iend = istart + Nxi_local - 1
    if ( (iend > Nxi) .OR. (myid .EQ. Nprocs-1) ) then
        iend = Nxi
    end if

    if (Parallel == 'x') then
        Nxstart = istart
        Nxend = iend
        Nystart = 1
        Nyend = Ny
        Nzstart = 1
        Nzend = Nz
    else if (Parallel == 'y') then
        Nystart = istart
        Nyend = iend
        Nxstart = 1
        Nxend = Nx
        Nzstart = 1
        Nzend = Nz
    else if (Parallel == 'z') then
        Nzstart = istart
        Nzend = iend
        Nxstart = 1
        Nxend = Nx
        Nystart = 1
        Nyend = Ny
    end if
    return

end subroutine DivideRegion

!------------------------------------------------------------------------
subroutine SendReceiveData(Nx, Ny, Nz, Nxstart, Nxend, Nystart, Nyend,&
        Nzstart, Nzend, Parallel, myid, Nprocs, ierr, U)
! Send data from all procs to 0 and return them in U
    use mpi
    implicit none

    integer Nx, Ny, Nz, &
        Nxstart,Nxend,Nystart,Nyend,Nzstart,Nzend
    integer i
    double precision U(Nx,Ny,Nz)

    integer myid, Nprocs, ierr, tag
    integer status(MPI_STATUS_SIZE)

    character Parallel*1

    if (myid /= 0) then
        tag = myid
        CALL MPI_SEND(U(Nxstart:Nxend,Nystart:Nyend,Nzstart:Nzend), &
            (Nxend-Nxstart+1)*(Nyend-Nystart+1)*(Nzend-Nzstart+1), &
            MPI_DOUBLE_PRECISION, 0, tag, MPI_COMM_WORLD, ierr)

    else if (myid == 0) then
        do i = 1, Nprocs-1
            CALL DivideRegion(Nx,Ny,Nz,Parallel,Nprocs,i, &
                Nxstart,Nxend,Nystart,Nyend,Nzstart,Nzend)

            tag = i
            CALL MPI_RECV(U(Nxstart:Nxend,Nystart:Nyend,Nzstart:Nzend), &
                (Nxend-Nxstart+1)*(Nyend-Nystart+1)*(Nzend-Nzstart+1), &
                MPI_DOUBLE_PRECISION, i, tag, MPI_COMM_WORLD, status, &
                ierr)
        end do
    end if
    return

end subroutine SendReceiveData

!------------------------------------------------------------------------
subroutine FindPotentials(Nx, Ny, Nz, Nxstart, Nxend, Nystart, Nyend, &
        Nzstart, Nzend, xmin, ymin, zmin, dxyz, Kp, parmod, &
        ByIMF, BzIMF, Tsygvxgse, Tsygvygse, Tsygvzgse, &
        pdyn, msw, ALindex, UseAL, U)

  implicit none

! Weimer's function needs this interface
  interface
     double precision function EpotVal(gLAT,gMLT)
       double precision gLAT, gMLT
     end function EpotVal
  end interface

  integer myid
  integer Nx, Ny, Nz, Nxstart, Nxend, Nystart, Nyend, &
      Nzstart, Nzend, Kp, L
  double precision U(Nx,Ny,Nz), dxyz(3), xmin, ymin, zmin
  real parmod(10), xi, yi, zi, dummy1(10), SPS, PSI, &
  dummy2(23)
  real xf, yf,zf, xx(20000), yy(20000), zz(20000), rr, &
      Xxgse, Yygse, Zzgse

  integer ii, jj, kk, mm
  double precision gLAT, gMLT, pi, NaN, arg
  real Reotu ! Radius of the Edge Of The Universe
  real Riono ! Radius of the ionospheric boundary
  real ByIMF, BzIMF, Tsygvxgse, Tsygvygse, Tsygvzgse, pdyn
  double precision msw, Tilt, Bt, angle, SWVel, SWDen, ALindex
  logical UseAL, J

#ifdef _T01_
  external T01_01, IGRF_GSW_08
#elif _TS05_
  external T04_s, IGRF_GSW_08
#elif _T96_
  external T96_01, IGRF_GSW_08
#endif

  ! This should give us the dipole tilt in the PSI variable
  COMMON /GEOPACK1/ dummy1, SPS, dummy2
  PSI = asin(SPS)

  pi = 3.14159265358979323846264338327950288419716d0
  arg = -1.0d0
  NaN = sqrt(arg)

  ! first some sort of initialisation of Weimer's model
  Tilt = dble(PSI*180/pi)
  Bt = sqrt(dble(ByIMF)**2.0d0 + dble(BzIMF)**2.0d0)
  ! angle is the IMF clock angle. 0 <= angle <= 360 degrees
  angle=(atan(dble(ByIMF)/dble(BzIMF)) - &
       (-1+sign(1.0d0,dble(BzIMF)))*pi/2.0d0)*180.0d0/pi;
  if (angle .lt. 0.0d0) then
     angle = angle + 360.0d0
  end if
  SWVel = sqrt(dble(Tsygvxgse)**2.0d0 + dble(Tsygvygse)**2.0d0 + &
       dble(Tsygvzgse)**2.0d0)
  ! Solar wind density from pdyn = nsw*msw*vsw^2
  SWDen = dble(pdyn) / (SWVel**2.0d0 * 1.0d6 * msw) * 1.0d-15
  call SetModel(angle,Bt,Tilt,SWVel,SWDen,ALindex,UseAL)
  ! then we can find the potentials

  ! radius of the end of the universe
  Reotu = real(sqrt( (dxyz(1)*dble(Nx-1))**2.0d0 + &
       (dxyz(2)*dble(Ny-1))**2.0d0 +(dxyz(3)*dble(Nz-1))**2.0d0 )/6371.2d3)
  ! radius of the ionosphere
  Riono = 1.0+85.0/6371.2

  if ( myid == 0) then
    write (*,*) 'Tracing field lines to compute potentials'
  end if

  do kk = Nzstart, Nzend
     write (*,*) '(', kk-Nzstart+1, '/', Nzend-Nzstart+1, ')'
     do jj = Nystart, Nyend
        do ii = Nxstart, Nxend
           xi = real( (xmin + (ii-1)*dxyz(1))/6371.2d3 )
           yi = real( (ymin + (jj-1)*dxyz(2))/6371.2d3 )
           zi = real( (zmin + (kk-1)*dxyz(3))/6371.2d3 )
            !trace to the north
           if (zi>0) then
#ifdef _T01_
           call TRACE_08(xi, yi, zi, -1.0, 1.0, 0.0001, &
                Reotu, Riono, Kp, parmod, &
                T01_01, IGRF_GSW_08, xf, yf, zf, xx, yy, &
                zz, L, 20000)
#elif _TS05_
           call TRACE_08(xi, yi, zi, -1.0, 1.0, 0.0001, &
                Reotu, Riono, Kp, parmod, &
                T04_s, IGRF_GSW_08, xf, yf, zf, xx, yy, &
                zz, L, 20000)
#elif _T96_
           call TRACE_08(xi, yi, zi, -1.0, 1.0, 0.0001, &
                Reotu, Riono, Kp, parmod, &
                T96_01, IGRF_GSW_08, xf, yf, zf, xx, yy, &
                zz, L, 20000)
#endif
           rr = sqrt(xf**2 + yf**2 +zf**2)
 !          if ( (rr>1.5) .or. (rr<0.5) )then
            ! trace to the south
            else
#ifdef _T01_
              call TRACE_08(xi, yi, zi, 1.0, 1.0, 0.0001, &
                   Reotu, Riono, Kp, parmod, &
                   T01_01, IGRF_GSW_08, xf, yf, zf, xx, yy, &
                   zz, L, 20000)
#elif _TS05_
              call TRACE_08(xi, yi, zi, 1.0, 1.0, 0.0001, &
                   Reotu, Riono, Kp, parmod, &
                   T04_s, IGRF_GSW_08, xf, yf, zf, xx, yy, &
                   zz, L, 20000)
#elif _T96_
              call TRACE_08(xi, yi, zi, 1.0, 1.0, 0.0001, &
                   Reotu, Riono, Kp, parmod, &
                   T96_01, IGRF_GSW_08, xf, yf, zf, xx, yy, &
                   zz, L, 20000)
#endif
              rr = sqrt(xf**2 + yf**2 +zf**2)
           end if
           ! If tracing failed set U to NaN, otherwise call Weimer.
           ! i should produce NaN for a real variable.
           if ( (rr>1.5) .or. (rr<0.5) )then
              U(ii, jj, kk) = NaN
              !Redefine the dipole for the SH in Weimer's model
           else
              call gswgse_08(xf,yf,zf,Xxgse,Yygse,Zzgse,1)
              if (Zzgse<0) then
                 Tilt=-dble(PSI*180/pi)
                 call SetModel(angle,Bt,Tilt,SWVel,SWDen,ALindex,UseAL)
              end if
              gLAT = abs(asin(dble(zf/rr)))*180.0d0/pi
              gMLT = acos(dble(-xf/sqrt(xf**2+yf**2)))*180.0d0/pi
              if ( yf>0 ) then
                 gMLT = 360.0d0 - gMLT
              end if
              gMLT = gMLT/15.0d0
              ! EpotVal gives us the potential in kV.
              ! We multiply by a thousand to get volts.
              U(ii, jj, kk) = EpotVal(gLAT,gMLT) * 1.0d3
           end if
        end do
     end do
  end do


end subroutine FindPotentials

!------------------------------------------------------------------------

recursive subroutine DumpScalarField(Field,Nx,Ny,Nz,fieldname,iteration, &
     attemptno, Nretries)

! No implicit varaibles
  implicit none

! Parameters
  integer Nx, Ny, Nz, iteration, attemptno, Nretries
  double precision Field(Nx,Ny,Nz)
  character fieldname*4

! Local variables
  integer i, j, k
  character filename*226, number*4

  attemptno = attemptno + 1

! Assemble filename, it is of the form 'particles_xxxx' where xxxx is
! the zero-padded iteration number (between 0 and 9999)

  write(number,fmt='(i4.4)') iteration

! Filename for scalar field
  filename = fieldname//number//'.ham.dat'

! Create and open file for writing scalar field
  open(unit=1,file=filename,status='replace',err=97)


  do k = 1, Nz
     do i = 1, Nx
        write (1,fmt='(E11.4)',err=98) (Field(i,j,k), j=1, Ny)
     end do
  end do

  close(1,err=99)

  return

! Error handling section
97 write (*,*) 'DumpScalarField: error in open statement'
  goto 100
98 write (*,*) 'DumpScalarField: error in write statement'
  close(1,err=99)
  goto 100
99 write (*,*) 'DumpScalarField: error in close statement'
100 if (attemptno<=Nretries) then
     call DumpScalarField(Field,Nx,Ny,Nz,fieldname,iteration, &
          attemptno, Nretries)
  end if

end subroutine DumpScalarField

!------------------------------------------------------------------------

recursive subroutine DumpVectorField(Field,Nx,Ny,Nz,fieldname,iteration, &
     attemptno, Nretries)


! No implicit varaibles
  implicit none

! Parameters
  integer Nx, Ny, Nz, iteration, attemptno, Nretries
  double precision Field(3,Nx,Ny,Nz)
  character fieldname*4

! Local variables
  integer i, j, k, p
  character filename*226, number*4

  attemptno = attemptno + 1


! Assemble filename, it is of the form 'particles_xxxx' where xxxx is
! the zero-padded iteration number (between 0 and 9999)

  write(number,fmt='(i4.4)') iteration

! Filename for electron position
  filename = fieldname//number//'.ham.dat'

! Create and open file for writing vector field
  open(unit=1,file=filename,status='replace',err=97)

  do p = 1,3
     do k = 1, Nz
        do i = 1, Nx
           write (1,fmt='(E11.4)',err=98) (Field(p,i,j,k), j=1, Ny)
        end do
     end do
  end do

  close(1,err=99)

  return

! Error handling section
97 write (*,*) 'DumpVectorField: error in open statement'
  goto 100
98 write (*,*) 'DumpVectorField: error in write statement'
  close(1,err=99)
  goto 100
99 write (*,*) 'DumpVectorField: error in close statement'
100 if (attemptno<=Nretries) then
     call DumpVectorField(Field,Nx,Ny,Nz,fieldname,iteration, &
          attemptno, Nretries)
  end if

end subroutine DumpVectorField

!------------------------------------------------------------------------

subroutine InterpolateVectorField(E, Nx, Ny, Nz, dxyz, xmin, ymin, zmin, &
     x0, y0, z0, E0)

  implicit none

  integer Nx, Ny, Nz
  double precision E(3,Nx,Ny,Nz), dxyz(3), xmin, ymin, zmin, &
       x0, y0, z0, E0(3)

  integer Ng(3), ff, ii, jj, kk
  double precision rest(3), drest(3), FieldE(2,2,2,3), F(8), dxdydz

  dxdydz = dxyz(1)*dxyz(2)*dxyz(3)

  Ng(1)    = int(dint((x0-xmin)/dxyz(1))) + 1
  rest(1)  = dmod((x0-xmin),dxyz(1))
  drest(1) = dxyz(1)-rest(1)
  Ng(2)    = int(dint((y0-ymin)/dxyz(2))) + 1
  rest(2)  = dmod((y0-ymin),dxyz(2))
  drest(2) = dxyz(2)-rest(2)
  Ng(3)    = int(dint((z0-zmin)/dxyz(3))) + 1
  rest(3)  = dmod((z0-zmin),dxyz(3))
  drest(3) = dxyz(3)-rest(3)

! If the grid is specified too small it is possible that the particle
! position is outside of the grid. If that happens we set E0=0 and return.
  if ( (minval(Ng)<=0) .or. (Ng(1)>=Nx) .or. (Ng(2)>=Ny) .or. (Ng(3)>=Nz) ) then
     E0 = 0.0d0
     return
  end if

! Find fields Fx, Fy, Fz  at eight closest gridpoints
  do ff = 1, 3
     do kk = 0, 1
        do jj = 0, 1
           do ii = 0, 1
              FieldE(ii+1,jj+1,kk+1,ff) = E(ff,Ng(1)+ii,Ng(2)+jj,Ng(3)+kk)
           end do
        end do
     end do
  end do

  F(1) = drest(1) * drest(2) * drest(3)
  F(2) =  rest(1) * drest(2) * drest(3)
  F(3) = drest(1) *  rest(2) * drest(3)
  F(4) = drest(1) * drest(2) *  rest(3)
  F(5) =  rest(1) *  rest(2) * drest(3)
  F(6) = drest(1) *  rest(2) *  rest(3)
  F(7) =  rest(1) * drest(2) *  rest(3)
  F(8) =  rest(1) *  rest(2) *  rest(3)

  do ff = 1, 3
     E0(ff) = FieldE(1,1,1,ff)*F(1) + FieldE(2,1,1,ff)*F(2) + &
          FieldE(1,2,1,ff)*F(3) + FieldE(1,1,2,ff)*F(4) + &
          FieldE(2,2,1,ff)*F(5) + FieldE(1,2,2,ff)*F(6) + &
          FieldE(2,1,2,ff)*F(7) + FieldE(2,2,2,ff)*F(8)

     E0(ff) = E0(ff)/dxdydz

  end do

end subroutine InterpolateVectorField

!------------------------------------------------------------------------

recursive subroutine DumpDump(U, E, Nx, Ny, Nz, attemptno, Nretries)


! No implicit variables
  implicit none

! Parameters
  integer iteration, Nx, Ny, Nz, attemptno, Nretries
  double precision U(Nx,Ny,Nz), E(3,Nx,Ny,Nz)

! Local variables
  character filename*237

  attemptno = attemptno + 1

! Filename for potential
  filename = 'dump_phi.ham.bin'

! Create and open file for writing potential
  open(unit=1,form='unformatted',file=filename,status='replace',err=97)
  write (1,err=98) U
  close(1,err=99)


! Filename for E-field
  filename = 'dump_Efi.ham.bin'

! Create and open file for writing potential
  open(unit=1,form='unformatted',file=filename,status='replace',err=97)
  write (1,err=98) E
  close(1,err=99)

  return

! Error handling section
97 write (*,*) 'DumpDump: error in open statement'
  goto 100
98 write (*,*) 'DumpDump: error in write statement'
  close(1,err=99)
  goto 100
99 write (*,*) 'DumpDump: error in close statement'
100 if (attemptno<=Nretries) then
     call DumpDump(U, E, Nx, Ny, Nz, attemptno, Nretries)
  end if

end subroutine DumpDump

!------------------------------------------------------------------------

recursive subroutine LoadDump(U, E, Nx, Ny, Nz, attemptno, Nretries)

! No implicit variables
  implicit none

! Parameters
  integer Nx, Ny, Nz, iter_start, attemptno, Nretries
  double precision U(Nx,Ny,Nz), E(3,Nx,Ny,Nz)

! Local variables
  character filename*237

  attemptno = attemptno + 1

! Filename for potential
  filename = 'dump_phi.ham.bin'

! Create and open file for reading potential
  open(unit=1,form='unformatted',file=filename,status='old',err=97)
  read(1,err=98) U
  close(1,err=99)


! Filename for E-field
  filename = 'dump_Efi.ham.bin'

! Create and open file for reading E-field
  open(unit=1,form='unformatted',file=filename,status='old',err=97)
  read(1,err=98) E
  close(1,err=99)

  return
! Error handling section
97 write (*,*) 'LoadDump: error in open statement'
  goto 100
98 write (*,*) 'LoadDump: error in read statement'
  close(1,err=99)
  goto 100
99 write (*,*) 'LoadDump: error in close statement'
100 if (attemptno<=Nretries) then
     call LoadDump(U, E, Nx, Ny, Nz, attemptno, Nretries)
  else
     stop
  end if

end subroutine LoadDump

!------------------------------------------------------------------------
subroutine TraceFieldLine(Pos,StepMax,Err,UpLim,DownLim,Kp,parmod, &
        FL, N_points)

    real Pos(3)

    real parmod(10), xf, yf, zf, Err, UpLim, DownLim, &
        StepMax

    real FL(40000,3)

    integer Kp, FL_points_dum, N_points

#ifdef _T01_
  external T01_01, IGRF_GSW_08
#elif _TS05_
  external T04_s, IGRF_GSW_08
#elif _T96_
  external T96_01, IGRF_GSW_08
#endif


    FL = 0.0

#ifdef _T96_
    CALL TRACE_08( &
        Pos(1), Pos(2), Pos(3), &
        -1.0, StepMax, Err, UpLim, DownLim, &
        Kp, parmod, T96_01, IGRF_GSW_08, &
        xf, yf, zf, &
        FL(N_points/2:1:-1,1), FL(N_points/2:1:-1,2), &
        FL(N_points/2:1:-1,3), FL_points_dum, N_points)
    CALL TRACE_08( &
        Pos(1), Pos(2), Pos(3), &
        1.0, StepMax, Err, UpLim, DownLim, &
        Kp, parmod, T96_01, IGRF_GSW_08, &
        xf, yf, zf, &
        FL(N_points/2+1:N_points,1), FL(N_points/2:N_points,2), &
        FL(N_points/2+1:N_points,3), FL_points_dum, N_points)
#elif _T01_
    CALL TRACE_08( &
        Pos(1), Pos(2), Pos(3), &
        -1.0, StepMax, Err, UpLim, DownLim, &
        Kp, parmod, T01_01, IGRF_GSW_08, &
        xf, yf, zf, &
        FL(N_points/2:1:-1,1), FL(N_points/2:1:-1,2), &
        FL(N_points/2:1:-1,3), FL_points_dum, N_points)
    CALL TRACE_08( &
        Pos(1), Pos(2), Pos(3), &
        1.0, StepMax, Err, UpLim, DownLim, &
        Kp, parmod, T01_01, IGRF_GSW_08, &
        xf, yf, zf, &
        FL(N_points/2+1:N_points,1), FL(N_points/2:N_points,2), &
        FL(N_points/2+1:N_points,3), FL_points_dum, N_points)
#elif _TS05_
    CALL TRACE_08( &
        Pos(1), Pos(2), Pos(3), &
        -1.0, StepMax, Err, UpLim, DownLim, &
        Kp, parmod, T04_s, IGRF_GSW_08, &
        xf, yf, zf, &
        FL(N_points/2:1:-1,1), FL(N_points/2:1:-1,2), &
        FL(N_points/2:1:-1,3), FL_points_dum, N_points)
    CALL TRACE_08( &
        Pos(1), Pos(2), Pos(3), &
        1.0, StepMax, Err, UpLim, DownLim, &
        Kp, parmod, T04_s, IGRF_GSW_08, &
        xf, yf, zf, &
        FL(N_points/2+1:N_points,1), FL(N_points/2:N_points,2), &
        FL(N_points/2+1:N_points,3), FL_points_dum, N_points)
#endif
    return
end subroutine TraceFieldLine

!------------------------------------------------------------------------
